/******************************************************************************
*                                                  
*  (c) copyright Freescale Semiconductor 2008
*  ALL RIGHTS RESERVED
*
*  File Name: ParseS19.c
*                                                                          
*  Purpose: This file is for a USB Mass-Storage Device bootloader.  This file 
*           has functions to read S19 file sent over USB, and parse s-records
*           to program to flash.
*                                                                          
*  Assembler:  Codewarrior for Microcontrollers V6.2
*                                            
*  Version:  1.3
*                                                                          
*                                                                          
*  Author: Derek Snell                             
*                                                                                       
*  Location: Indianapolis, IN. USA                                            
*                                                                                  
* UPDATED HISTORY:
*
* REV   YYYY.MM.DD  AUTHOR        DESCRIPTION OF CHANGE
* ---   ----------  ------        --------------------- 
* 1.3   2009.10.19  Derek Snell   Reduced bootloader RAM requirements for JM16
*                                 made srec_buffer a global array to remove from stack
* 1.2   2009.01.13  Derek Snell   Added linker SEGMENTs for S08 version
*                                 Changed compiler directives for programming flash for S08
* 1.1   2008.09.17  Derek Snell   Updated to give S19 address error in status
* 1.0   2008.06.10  Derek Snell   Initial version
* 
*
******************************************************************************/                                                                        
/* Freescale  is  not  obligated  to  provide  any  support, upgrades or new */
/* releases  of  the Software. Freescale may make changes to the Software at */
/* any time, without any obligation to notify or provide updated versions of */
/* the  Software  to you. Freescale expressly disclaims any warranty for the */
/* Software.  The  Software is provided as is, without warranty of any kind, */
/* either  express  or  implied,  including, without limitation, the implied */
/* warranties  of  merchantability,  fitness  for  a  particular purpose, or */
/* non-infringement.  You  assume  the entire risk arising out of the use or */
/* performance of the Software, or any systems you design using the software */
/* (if  any).  Nothing  may  be construed as a warranty or representation by */
/* Freescale  that  the  Software  or  any derivative work developed with or */
/* incorporating  the  Software  will  be  free  from  infringement  of  the */
/* intellectual property rights of third parties. In no event will Freescale */
/* be  liable,  whether in contract, tort, or otherwise, for any incidental, */
/* special,  indirect, consequential or punitive damages, including, but not */
/* limited  to,  damages  for  any loss of use, loss of time, inconvenience, */
/* commercial loss, or lost profits, savings, or revenues to the full extent */
/* such  may be disclaimed by law. The Software is not fault tolerant and is */
/* not  designed,  manufactured  or  intended by Freescale for incorporation */
/* into  products intended for use or resale in on-line control equipment in */
/* hazardous, dangerous to life or potentially life-threatening environments */
/* requiring  fail-safe  performance,  such  as  in the operation of nuclear */
/* facilities,  aircraft  navigation  or  communication systems, air traffic */
/* control,  direct  life  support machines or weapons systems, in which the */
/* failure  of  products  could  lead  directly to death, personal injury or */
/* severe  physical  or  environmental  damage  (High  Risk Activities). You */
/* specifically  represent and warrant that you will not use the Software or */
/* any  derivative  work of the Software for High Risk Activities.           */
/* Freescale  and the Freescale logos are registered trademarks of Freescale */
/* Semiconductor Inc.                                                        */ 
/*****************************************************************************/


#include "derivative.h" /* include peripheral declarations */
#include "ParseS19.h"
#include "SCSI_Process.h"
#include "Bootloader_Headers.h"


#ifdef _MCF51JM128_H
	#include <hidef.h>
#endif
	
/*****************************************************************************
 * External references.
 *****************************************************************************/
 extern byte  BlockWriteDone;



/*****************************************************************************
 * Module variables.
 *****************************************************************************/
 unsigned char *SRecCharPointer;
 unsigned char *SRecBufferEndAddress;
 unsigned char S19FileDone;
 dword S19Address; 
 unsigned char srec_buffer[252];

 
/*****************************************************************************
 * Public memory declarations
 *****************************************************************************/
 extern unsigned char BootloaderStatus;
 extern unsigned char FlashErased;
 extern byte  vCSWResult;         // CSW result
 extern byte vCBWBuf_flag;
 extern byte vEP2Data[MSD_BUFFER_SIZE];
 extern volatile unsigned char FLASHPGM[59];   


/*****************************************************************************
 * Function predefinitions.
 *****************************************************************************/
 void GetUSBOutFileData(void);
 void Send_CSW(void);
 void PollUSB (void);
 void SCSI_Process(void);
 unsigned char CheckAddressValid(dword Address);
 void EraseFlash(void);
 #ifdef S08_CORE 
    unsigned char Flash_Prog(unsigned int dst_addr, unsigned int data_addr, unsigned char length) ; 
 #endif S08_CORE


/*********************************************************
* Name: GetHexValue
*
* Desc: Converts ASCII character to hex value 
*
* Parameter: text - ASCII to convert 
*
* Return: unsigned char, hex value of character
*             
**********************************************************/
unsigned char GetHexValue (unsigned char text)
{
    switch (text)
    {
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            return (byte)(text - '0');
        case 'A':
        case 'a':
            return 10;
        case 'B':
        case 'b':
            return 11;
        case 'C':
        case 'c':
            return 12;
        case 'D':
        case 'd':
            return 13;
        case 'E':
        case 'e':
            return 14;
        case 'F':
        case 'f':
            return 15;
        default:
            return 0xFF;
    }
}


/*********************************************************
* Name: GetUSBchar
*
* Desc: Gets next character out of buffer used for USB data
*
* Parameter: None 
*
* Return: unsigned char, Next character in buffer
*             
**********************************************************/
unsigned char GetUSBchar (void)
{
    unsigned char c;
   
    c = *SRecCharPointer;
    SRecCharPointer++;
    
    if(SRecCharPointer == SRecBufferEnd) {    // Pointer at end of USB buffer
        if(!BlockWriteDone) {                 // if more USB data coming in same write transfer
            GetUSBOutFileData();              // Put USB data into buffer
        } if(BlockWriteDone) {
            vCSWResult = kCSWPass;
            Send_CSW();                       // finish current USB transfer
          
            if(!S19FileDone){                 // Need to get next USB transfer started
                while(BlockWriteDone) {       // Wait to receive new block of write data
                    PollUSB();                // Wait for USB data to come and fill buffer
                    if(vCBWBuf_flag==1)
                    {
                        SCSI_Process();
                        vCBWBuf_flag=0;
                    }
                }
            }
        }
    SRecCharPointer = SRecBufferAddress;    // reset pointer to beginning of buffer
    }
    
    return c;
}


/*********************************************************
* Name: GetS
*
* Desc: This routine skips all characters until an `S' is received.
*       It then returns the next character.
*
* Parameter: None 
*
* Return: unsigned char, Next character after 'S'
*             
**********************************************************/
unsigned char GetS (void)
{
    unsigned char c;

    for(;;)
    {
        c = GetUSBchar();
        if (c == 'S')
            break;
    }

    /* Get type of S-record */
    return GetUSBchar();
}

/*********************************************************
* Name: GetSpair
*
* Desc: Gets pair of characters, and converts to hex byte
*
* Parameter: None 
*
* Return: unsigned char, converted hex byte
*             
**********************************************************/
unsigned char GetSpair (void)
{
    unsigned char ch;
    unsigned char upper;

    ch = GetUSBchar();
    upper = (unsigned char)(GetHexValue(ch));
    
    if(upper == 0xFF) {           // Not a proper S19 file
        S19FileDone = TRUE;
    } else
        upper = (byte)(upper << 4);
    
    ch = GetUSBchar();
    ch = (unsigned char)(GetHexValue(ch));
    if(ch == 0xFF) {           // Not a proper S19 file
        S19FileDone = TRUE;
    } 
    
    return (byte)(upper | ch);
}




/*********************************************************
* Name: ParseSRec
*
* Desc: Read from buffer of S-Record, and parse for programming 
*
* Parameter: None 
*
* Return: None
*             
**********************************************************/
void ParseS19(void) {

 	  unsigned char length, checksum, data, i, temp, offset;
    unsigned char type;
    
    SRecCharPointer = SRecBufferAddress;
    
    while (!S19FileDone)
    {
        // Get start of S-record 
        type = GetS();
        
        // Get record length 
        length = GetSpair();
        if(S19FileDone) {       // not a valid S19 file
            BootloaderStatus = BootloaderS19Error;
            return;
        }
        checksum = length;

        // Take appropriate action 
        switch (type)
        {
            case '1':
            case '2':
            case '3':
                S19Address = (dword) NULL;
                type -= '0';
                for (i = 0; i <= type ; i++)
                {
                    // Formulate address
                    // Address needs to be word aligned for successful flash program 
                    data = GetSpair();
                    if(S19FileDone) {       // not a valid S19 file
                        BootloaderStatus = BootloaderS19Error;
                        return;
                    }
                    S19Address = (S19Address << 8) | data;
                    
                    // Maintain 8-bit checksum 
                    checksum = (unsigned char)((data + checksum) & 0x00FF);
                    length--;
                }
                
                if (CheckAddressValid(S19Address))
                {
  
                    // If flash not erased, erase flash
                    if(!FlashErased){
                        EraseFlash();
                        if(BootloaderStatus == BootloaderFlashError) {
                            // Abort S19 download
                            S19FileDone = TRUE;
                            return;
                        }
                        FlashErased = TRUE;
                    }
                                  
                    
                    // 32-bit cores program flash in 32-bit words
                    // Therefore S19 address needs to be adjusted to be word aligned
                    #ifdef CF_CORE
                      // Pad beginning of srec_buffer if address not word aligned
                      offset = (byte) (S19Address & 0x0003);
                      S19Address = (dword) (S19Address & 0xFFFFFFFC);
                      length += offset;
                      for (i = 0; i < offset; i++) {
                          srec_buffer[i] = 0xFF; 
                      }
                    
                    
                    // 8-bit cores program flash in 8-bit bytes
                    // S19 address does not need to be adjusted
                    #elif defined S08_CORE
                      offset = 0;
                    
                    #else
                      #error "No Core Defined"
                    #endif  CF_CORE
                    
                    // Get data and put into srec_buffer 
                    for (i = offset; i < (length - 1); i++)
                    {
                        srec_buffer[i] = GetSpair();
                        if(S19FileDone) {       // not a valid S19 file
                            BootloaderStatus = BootloaderS19Error;
                            return;
                        }
                          
                    }

                    // Calculate checksum 
                    for (i = offset; i < (length - 1); i++)
                    {
                        checksum = (unsigned char)((srec_buffer[i] + checksum) & 0x00FF);
                    }
                    
                    // Get checksum byte 
                    data = GetSpair();
                    if(S19FileDone) {       // not a valid S19 file
                        BootloaderStatus = BootloaderS19Error;
                        return;
                    }
                    
                    if (((data - ~checksum) & 0x00FF) != 0)
                    {
                        BootloaderStatus = BootloaderS19Error;
                        S19FileDone = TRUE;
                        return;
                    }
                    
                    // For 32-bit cores Flash_Prog writes 32-bit words, not bytes.
                    // if last 32-bit word in s-record is not complete, finish word
                    #ifdef CF_CORE
                      if((i & 0x0003) != 0x0000) {    // 32-bit word not complete
                          srec_buffer[i++] = 0xFF;         // pad end of word 
                          srec_buffer[i++] = 0xFF;         // pad end of word 
                          srec_buffer[i++] = 0xFF;         // pad end of word 
                      }
                    #endif  CF_CORE
                    // NOTE: 8-bit core does not need to pad the end

                    // Write buffered data to Flash 
                    #ifdef S08_CORE
                      if(((S19Address >= MIN_FLASH2_ADDRESS) && (S19Address <= MAX_FLASH2_ADDRESS)) || ((S19Address >= MIN_FLASH1_ADDRESS) && (S19Address < FLASH_PROTECTED_ADDRESS))) {
                    #elif defined CF_CORE
                      if((S19Address > FLASH_PROTECTED_ADDRESS) && (S19Address <= MAX_FLASH1_ADDRESS)) {
                    #endif S08_CORE
                        
                        // call flash program
                        #ifdef S08_CORE	// call this way for S08_CORE
                          temp = Flash_Prog((word)(S19Address & 0xFFFF), (word) &srec_buffer, i) ;
                        #elif defined  V1_CORE	// call this way for V1_CORE
                        	temp = (unsigned char) Flash_Prog(S19Address, (dword) &srec_buffer, (byte)(i >> 2));
                        #elif defined V2_CORE // call this way for V2_CORE
                        	temp = (unsigned char) Flash_Prog(S19Address, (dword) &srec_buffer, (i >> 2));
                        #else
                          #error "No Core Selected"
                        #endif
                        	
                        if(gFlashError == temp){
                            BootloaderStatus = BootloaderFlashError;
                            return;
                        }
                    }
                                
                }

                else    // S-Record points to invalid address
                {
                    BootloaderStatus = BootloaderS19Error;
                    return;
                }

                break;
            case '7':
            case '8':
            case '9':
                S19Address = (dword) NULL; 
                type = (unsigned char)(type - '0');
                type = (unsigned char)(10 - type);
                
                // Get Address
                for (i = 0; i <= type ; i++)
                {
                    data = GetSpair();
                    if(S19FileDone) {       // not a valid S19 file
                        BootloaderStatus = BootloaderS19Error;
                        return;
                    }
                    checksum = (unsigned char)((data + checksum) & 0x00FF);
                    S19Address = (S19Address << 8) | data;
                    length--;
                }
                
                // Get Data
                while (length-- > 1)
                {
                    data = GetSpair();
                    if(S19FileDone) {       // not a valid S19 file
                        BootloaderStatus = BootloaderS19Error;
                        return;
                    }
                    checksum = (unsigned char)((data + checksum) & 0x00FF);
                }

                // Read checksum value
                data = GetSpair();
                if(S19FileDone) {       // not a valid S19 file
                    BootloaderStatus = BootloaderS19Error;
                    return;
                }

                if (((data - ~checksum) & 0x00FF) != 0)
                {
                    BootloaderStatus = BootloaderS19Error;
                    S19FileDone = TRUE;
                    return;
                }
                else // File completely read successfully
                {
                    BootloaderStatus = BootloaderSuccess;
                    S19FileDone = TRUE;
                    return;
                }
                
                break;
            case '0':
            case '4':
            case '5':
            case '6':
            default:
                break;
        }
    }
    return;    
}

